/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2005 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "$Id: mx_lx.c,v 1.16.2.2 2006/09/20 17:49:28 loic Exp $";

#include "mx_arch.h"
#include "mx_instance.h"
#include "mx_malloc.h"
#include "mx_misc.h"
#include "mx_peer.h"
#include "mx_pio.h"
#include "mcp_config.h"
#include "mx_stbar.h"
#include "mx_util.h"


/* Hack to avoid clashes between system header files and zlib.h on
   for the declaration of crc32() on some OSes */
#define MX_INSTANCE_C
#ifdef EXPORT
#undef EXPORT
#endif


/* put the board in reset via PCI config space */
static void
mx_board_reset_on(mx_instance_state_t *is)
{
  mx_write_lanai_ctrl_bit(is, MX_LX_CTRL_IO_RESET_ON_BIT);
  MX_STBAR();
}

/* put the LANai in reset via PCI config space */
static void
mx_lanai_reset_on(mx_instance_state_t *is)
{
  mx_write_lanai_ctrl_bit(is,  MX_LX_CTRL_CPU_RESET_ON_BIT);
  MX_STBAR();
}

/* take the board out of reset via PCI config space */
static void
mx_board_reset_off(mx_instance_state_t *is)
{
  mx_write_lanai_ctrl_bit(is, MX_LX_CTRL_IO_RESET_OFF_BIT);
  MX_STBAR();
}

/* take the LANai out of reset via PCI config space */
static void
mx_lanai_reset_off(mx_instance_state_t *is)
{
  mx_write_lanai_ctrl_bit(is,  MX_LX_CTRL_CPU_RESET_OFF_BIT);
  MX_STBAR();
}

/* enable interrupts */
static void
mx_lx_int_enable(mx_instance_state_t *is)
{
  mx_write_lanai_ctrl_bit(is, MX_LX_CTRL_PCI_INT_ENABLE_ON_BIT);
  MX_STBAR();
}

/* disable interrupts */
static void
mx_lx_int_disable(mx_instance_state_t *is)
{
  mx_write_lanai_ctrl_bit(is, MX_LX_CTRL_PCI_INT_ENABLE_OFF_BIT);
  MX_STBAR();
}

static void
mx_lx_park_board(mx_instance_state_t *is)
{
  /* Park the board so that the CPU is frozen, and the hardware
     will drop packets */

  mx_lanai_reset_on(is);
  mx_board_reset_on(is);
  mx_spin(20000);
  mx_board_reset_off(is);

}

static int
mx_lx_detect_parity(mx_instance_state_t *is)
{
  return mx_read_lanai_special_reg_u32(is, lx.ISR) & MX_LX_PARITY_INT;
}

#ifndef MX_ARCH_ZLIB_INFLATE
#include "zlib.h"

#define mx_zlib_inflate inflate
#define mx_zlib_inflateEnd inflateEnd

static voidpf
mx_zlib_calloc(voidpf opaque, uInt items, uInt size)
{
  return mx_kmalloc(items * size, MX_MZERO|MX_WAITOK);
}

static void
mx_zlib_free(voidpf opaque, voidpf addr)
{
  mx_kfree(addr);
}

static inline int
mx_zlib_inflateInit(z_stream *zs)
{
  zs->zalloc = mx_zlib_calloc;
  zs->zfree = mx_zlib_free;
  zs->opaque = 0;
  return inflateInit(zs);
}
#endif /* MX_ARCH_ZLIB_INFLATE */

int
mx_load_mcp(mx_instance_state_t *is, void *inflate_buffer, int limit, int *actual_len)
{
  z_stream zs;
  int rv, status;
  unsigned char *mcp;
  int mcp_len;
  unsigned int parity_critical_start, parity_critical_end;

  status = 0;

  status = mx_mcpi.select_mcp(is->id, is->board_type, &mcp, &mcp_len,
			      &parity_critical_start,
			      &parity_critical_end);
  if (status || mcp == NULL) {
    MX_WARN(("mx%d: Could not find mcp for type %d card (rev %d)\n", 
	     is->id, is->board_type, is->pci_rev));
    return ENODEV;
  }

  is->parity_critical_start = (uintptr_t)&is->lanai.sram[parity_critical_start];
  is->parity_critical_end = (uintptr_t)&is->lanai.sram[parity_critical_end];

  /* align them on 8 byte boundaries */
  is->parity_critical_start &= ~(uintptr_t)7;
  is->parity_critical_end = (is->parity_critical_end + (uintptr_t)7) & (uintptr_t)~7;

  rv = mx_zlib_inflateInit(&zs);
  if (rv != Z_OK) {
    MX_WARN(("Fatal error %d in inflateInit\n", rv));
    return ENXIO;
  }

  zs.next_in = mcp;
  zs.avail_in = mcp_len;
  zs.next_out = (unsigned char *) inflate_buffer;
  zs.avail_out = limit;

  rv = mx_zlib_inflate(&zs, Z_FINISH);
  if (rv != Z_STREAM_END || zs.avail_in != 0 || zs.avail_out == 0) {
    MX_NOTE(("fatal error in inflate - mcp too big   err %d?\n", rv));
    MX_NOTE(("   lanai sram space originally    = %d, limit=%d\n",
	       is->sram_size, limit));
    MX_NOTE(("   bytes remaining to uncompress  = %d\n",
	       zs.avail_in));
    MX_NOTE(("   Available space for decompress = %d\n",
	       zs.avail_out));
    MX_NOTE(("   inflate error msg %s\n", zs.msg));
    MX_NOTE(("   inflate error char %x\n",
	       *(unsigned char *) zs.next_in));
    MX_NOTE(("   inflate error char %x\n",
	       *(unsigned char *) (zs.next_in - 1)));
    MX_NOTE(("   inflate error char %x\n",
	       *(unsigned char *) (zs.next_in + 1)));
    MX_NOTE(("   inflate error total %ld\n", zs.total_in));
    status = ENXIO;
    goto abort_with_inflate_init;
  }

  /* Make sure we don't copy more than was uncompressed. */
  if (zs.total_out < limit)
    limit = zs.total_out;

  *actual_len = limit;
  mx_zlib_inflateEnd(&zs);
  return 0;

 abort_with_inflate_init:
  mx_zlib_inflateEnd(&zs);

  return status;
  
}

static void
mx_lx_check_parity(mx_instance_state_t *is, const char *stage)
{
  if (mx_read_lanai_special_reg_u32(is, lx.ISR) & MX_LX_PARITY_INT) {
    mx_write_lanai_isr(is, MX_LX_PARITY_INT);
    MX_WARN(("%s, parity was set and was cleared\n", stage));
  }
}

static int
mx_lx_init_board(mx_instance_state_t *is)
{
  volatile char *eeprom_strings;
  int usable_sram_len, status;
  int mcp_len;
  void *img;

  usable_sram_len = is->sram_size - MX_EEPROM_STRINGS_LEN;

  MX_DEBUG_PRINT (MX_DEBUG_BOARD_INIT, ("Putting board into reset\n"));
  mx_board_reset_on(is);

  MX_DEBUG_PRINT (MX_DEBUG_BOARD_INIT, ("Putting LANai into reset\n"));
  mx_lanai_reset_on(is);
  mx_spin(20000);

  MX_DEBUG_PRINT (MX_DEBUG_BOARD_INIT, ("Taking board out of reset\n"));
  mx_board_reset_off(is);
  mx_spin(20000);
  mx_lx_check_parity(is, "After out-of-reset");
  /* save the eeprom strings in case we overrite them by accident */
  eeprom_strings = (volatile char *)is->lanai.sram + usable_sram_len;
  bcopy((char *)eeprom_strings, is->lanai.eeprom_strings, 
	MX_EEPROM_STRINGS_LEN);
  is->lanai.eeprom_strings[MX_EEPROM_STRINGS_LEN - 1] = 0;
  is->lanai.eeprom_strings[MX_EEPROM_STRINGS_LEN - 2] = 0;

  mx_lx_check_parity(is, "After string-spec-read");
  mx_parse_eeprom_strings(is);
  if (!is->lanai.product_code || !is->lanai.part_number || !is->mac_addr_string) {
    char *s;
    MX_WARN(("EEPROM Dump:\n"));
    for (s = is->lanai.eeprom_strings; *s; s += strlen(s) + 1) {
      MX_WARN(("\t%s\n", s));
    }
    MX_WARN(("Incomplete EEPROM, please contact help@myri.com\n"));
    return EIO;
  }
  if ((is->lanai.part_number && strcmp(is->lanai.part_number, "09-02887") == 0) 
      || (is->lanai.product_code && 
	  strcmp(is->lanai.product_code, "M3S-PCIXD-4-I") == 0)) {
    MX_WARN(("PC=%s, PN=%s is not supported\n", 
	     is->lanai.product_code, is->lanai.part_number));
    return EIO;
  }
  mx_pio_bzero((char *)is->lanai.sram, usable_sram_len);
  MX_STBAR();
  mx_spin(20000);
  MX_DEBUG_PRINT (MX_DEBUG_BOARD_INIT, ("Loading MCP\n"));
  img = mx_kmalloc(usable_sram_len, MX_MZERO | MX_WAITOK);
  if (!img) {
    MX_WARN(("Could not allocate buffer to inflate mcp\n"));
    return ENOMEM;
  }
  status = mx_load_mcp(is, img, usable_sram_len, &mcp_len);
  if (status != 0) {
    MX_WARN(("mx%d: Could not load mcp\n", is->id));
    mx_kfree(img);
    return status;
  }
  /* Copy the inflated firmware to NIC SRAM. */
  mx_pio_bcopy_write(img, (char*)is->lanai.sram, mcp_len);
  mx_kfree(img);
  mx_spin(20000);
  MX_STBAR();
  
  /* make sure that REQ_ACK_1 is clear, otherwise MCP will start 
     with unintialized globals */
  if (mx_read_lanai_special_reg_u32(is, lx.ISR) & MX_LX_REQ_ACK_1)
    mx_write_lanai_isr(is, MX_LX_REQ_ACK_1);
  
  /* make sure that REQ_ACK_0 is clear, otherwise lanai could start
     interrupt storm */
  if (mx_read_lanai_special_reg_u32(is, lx.ISR) & MX_LX_REQ_ACK_0)
    mx_write_lanai_isr(is, MX_LX_REQ_ACK_0);
  MX_STBAR();

  MX_DEBUG_PRINT (MX_DEBUG_BOARD_INIT, ("Taking LANAI out of reset\n"));
  mx_lanai_reset_off(is);
  mx_lx_int_enable(is);
  MX_STBAR();
  return 0;
}

static void
mx_lx_claim_interrupt(mx_instance_state_t *is)
{
  mx_write_lanai_isr(is, MX_LX_REQ_ACK_0);
}

static int
mx_lx_might_be_interrupting(mx_instance_state_t *is)
{
  if (is->using_msi) {
    return MX_CPU_powerpc64 ? MX_INTR_MAYBE : MX_INTR_ACTIVE;
  } else if (MX_CPU_x86) {
    /* avoid serverworks PIO read bug */
    return MX_INTR_MAYBE;
  } else {
    return (mx_read_lanai_special_reg_u32(is, lx.ISR) & MX_LX_REQ_ACK_0) ?
      MX_INTR_ACTIVE : MX_INTR_NONE;
  }
}

/* map the specials and SRAM.*/

static int
mx_lx_map_board(mx_instance_state_t *is)
{
  uint32_t dma_config;

  is->lanai.special_regs = mx_map_pci_space(is, 0, MX_LX_SPECIAL_OFFSET,
					   MX_LX_SPECIAL_LEN);
  if (is->lanai.special_regs == NULL) {
    goto abort_with_nothing;
  }
  is->specials_size = MX_LX_SPECIAL_LEN;

  is->lanai.control_regs = mx_map_pci_space(is, 0, MX_LX_CONTROL_OFFSET,
					   MX_LX_CONTROL_LEN);
  if (is->lanai.control_regs == NULL) {
    goto abort_with_special_regs;
  }

  dma_config = mx_read_lanai_special_reg_u32(is, lx.DMA_CONFIG);
  is->sram_size = dma_config & MX_LX_PCI_OFFSET;

  if ((is->sram_size < 1  * 1024 * 1024) ||
      (is->sram_size > 16 * 1024 * 1024)) {
    MX_WARN (("mx%d: Nonsensical sram size! (0x%x)\n", is->id, is->sram_size));
    goto abort_with_control_regs;
  }

  is->lanai.sram = mx_map_pci_space(is, 0, 0, is->sram_size);
  if (is->lanai.sram == 0) {
    goto abort_with_control_regs;
  }

  MX_DEBUG_PRINT (MX_DEBUG_BOARD_INIT, 
		    ("dma_config = 0x%x\n", dma_config));

  return 0;

 abort_with_control_regs:
  mx_unmap_io_space(is, MX_LX_CONTROL_LEN, (void*)is->lanai.control_regs);
  is->lanai.control_regs = NULL;
 abort_with_special_regs:
  mx_unmap_io_space(is, MX_LX_SPECIAL_LEN, is->lanai.special_regs);
  is->lanai.special_regs = NULL;
 abort_with_nothing:
  return ENODEV;
}

static void
mx_lx_unmap_board(mx_instance_state_t *is)
{
  mx_unmap_io_space(is, is->sram_size, (void*)is->lanai.sram);
  is->lanai.sram = NULL;
  mx_unmap_io_space(is, MX_LX_CONTROL_LEN, (void*)is->lanai.control_regs);
  is->lanai.control_regs = NULL;
  mx_unmap_io_space(is, MX_LX_SPECIAL_LEN, is->lanai.special_regs);
  is->lanai.special_regs = NULL;
}


static void
mx_lx_dump_registers(mx_instance_state_t *is,
		     uint32_t *reg, uint32_t *cnt)
{
  *cnt = 56;
  if (is->num_ports == 2)
    (*cnt) += 26;

  if (reg == NULL)
    return;

  /* HACK -- ISR must always be first register read */
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.ISR); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.ISR); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.AISR); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.AISR);reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.AISR_ON); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.AISR_ON); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.AISR_OFF); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.AISR_OFF); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.DISPATCH_INDEX); reg++;
  *reg = mx_read_lanai_special_reg_u16(is, lx.DISPATCH_INDEX); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.CPUC); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.CPUC); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.RTC); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.RTC); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.IT0); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.IT0); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.IT1); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.IT1); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.IT2); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.IT2); reg++;

  *reg = offsetof(mx_lanai_special_registers_t, read.lx.DMA_CONFIG); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.DMA_CONFIG); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.DMA0_COUNT); reg++;
  *reg = mx_read_lanai_special_reg_u8(is, lx.DMA0_COUNT); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.DMA1_COUNT); reg++;
  *reg = mx_read_lanai_special_reg_u8(is, lx.DMA1_COUNT); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.DMA2_COUNT); reg++;
  *reg = mx_read_lanai_special_reg_u8(is, lx.DMA2_COUNT); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.DMA3_COUNT); reg++;
  *reg = mx_read_lanai_special_reg_u8(is, lx.DMA3_COUNT); reg++;


  /* P0 RECV */
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P0_RPL_CONFIG); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.P0_RPL_CONFIG); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P0_RB); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.P0_RB); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P0_RPL_INDEX); reg++;
  *reg = mx_read_lanai_special_reg_u16(is, lx.P0_RPL_INDEX); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P0_RECV_COUNT); reg++;
  *reg = mx_read_lanai_special_reg_u16(is, lx.P0_RECV_COUNT); reg++;

  /* P0 Alternate RECV */
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P0_A_RPL_CONFIG); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.P0_A_RPL_CONFIG); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P0_A_RB); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.P0_A_RB); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P0_A_RPL_INDEX); reg++;
  *reg = mx_read_lanai_special_reg_u16(is, lx.P0_A_RPL_INDEX); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P0_A_RECV_COUNT); reg++;
  *reg = mx_read_lanai_special_reg_u16(is, lx.P0_A_RECV_COUNT); reg++;


  /* P0 Send */
  
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P0_SEND_FREE_COUNT); reg++;
  *reg = mx_read_lanai_special_reg_u8(is, lx.P0_SEND_FREE_COUNT); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P0_SEND_FREE_LIMIT); reg++;
  *reg = mx_read_lanai_special_reg_u8(is, lx.P0_SEND_FREE_LIMIT); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P0_SEND_COUNT); reg++;
  *reg = mx_read_lanai_special_reg_u8(is, lx.P0_SEND_COUNT); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P0_PAUSE_COUNT); reg++;
  *reg = mx_read_lanai_special_reg_u8(is, lx.P0_PAUSE_COUNT); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P0_SEND_CONTROL); reg++;
  *reg = mx_read_lanai_special_reg_u8(is, lx.P0_SEND_CONTROL); reg++;

  if (is->num_ports == 1)
    return;
  
  
  /* P1 RECV */
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P1_RPL_CONFIG); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.P1_RPL_CONFIG); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P1_RB); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.P1_RB); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P1_RPL_INDEX); reg++;
  *reg = mx_read_lanai_special_reg_u16(is, lx.P1_RPL_INDEX); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P1_RECV_COUNT); reg++;
  *reg = mx_read_lanai_special_reg_u16(is, lx.P1_RECV_COUNT); reg++;

  /* P1 Alternate RECV */
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P1_A_RPL_CONFIG); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.P1_A_RPL_CONFIG); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P1_A_RB); reg++;
  *reg = mx_read_lanai_special_reg_u32(is, lx.P1_A_RB); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P1_A_RPL_INDEX); reg++;
  *reg = mx_read_lanai_special_reg_u16(is, lx.P1_A_RPL_INDEX); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P1_A_RECV_COUNT); reg++;
  *reg = mx_read_lanai_special_reg_u16(is, lx.P1_A_RECV_COUNT); reg++;


  /* P1 Send */
  
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P1_SEND_FREE_COUNT); reg++;
  *reg = mx_read_lanai_special_reg_u8(is, lx.P1_SEND_FREE_COUNT); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P1_SEND_FREE_LIMIT); reg++;
  *reg = mx_read_lanai_special_reg_u8(is, lx.P1_SEND_FREE_LIMIT); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P1_SEND_COUNT); reg++;
  *reg = mx_read_lanai_special_reg_u8(is, lx.P1_SEND_COUNT); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P1_PAUSE_COUNT); reg++;
  *reg = mx_read_lanai_special_reg_u8(is, lx.P1_PAUSE_COUNT); reg++;
  *reg = offsetof(mx_lanai_special_registers_t, read.lx.P1_SEND_CONTROL); reg++;
  *reg = mx_read_lanai_special_reg_u8(is, lx.P1_SEND_CONTROL); reg++;
}

static void
mx_lx_get_freq(mx_instance_state_t *is)
{
  uint32_t pci_clock;
  int status;
  
  status = mx_mcpi.get_param(is->id, is->lanai.sram, "clock_freq",
			     &is->cpu_freq);
  
  if (status || is->cpu_freq == 0) {
    MX_WARN(("mx%d: Failed to read LANai clock frequency (%d, %d)\n", 
	     is->id, status, is->cpu_freq));
    return;
  }
  
  pci_clock = (uint32_t)mx_read_lanai_special_reg_u16 (is, lx.PCI_CLOCK);
  is->pci_freq = pci_clock;
}

static void
mx_lx_write_kreq(mx_instance_state_t *is, mcp_kreq_t *kreq)
{
  int kreqq_index;
  
  kreqq_index = is->kreqq_submitted & (is->kreqq_max_index);
  mx_always_assert((is->kreqq_submitted - is->kreqq_completed)
                   <= is->kreqq_max_index);
  is->kreqq_submitted++;
  mx_pio_memcpy(is->kreqq[kreqq_index].int64_array, kreq->int64_array, 
	      sizeof(mcp_kreq_t), MX_PIO_LAST|MX_PIO_FLUSH);
}

mx_board_ops_t mx_lx_ops;

void
mx_lx_init_board_ops(void)
{
  mx_lx_ops.init		= mx_lx_init_board;
  mx_lx_ops.map			= mx_lx_map_board;
  mx_lx_ops.unmap		= mx_lx_unmap_board;
  mx_lx_ops.claim_interrupt	= mx_lx_claim_interrupt;
  mx_lx_ops.might_be_interrupting = mx_lx_might_be_interrupting;
  mx_lx_ops.enable_interrupt	= mx_lx_int_enable;
  mx_lx_ops.disable_interrupt	= mx_lx_int_disable;
  mx_lx_ops.park		= mx_lx_park_board;
  mx_lx_ops.detect_parity_error	= mx_lx_detect_parity;
  mx_lx_ops.dump_registers	= mx_lx_dump_registers;
  mx_lx_ops.get_freq		= mx_lx_get_freq;
  mx_lx_ops.write_kreq		= mx_lx_write_kreq;
}
